#!/usr/bin/env python3
import os, subprocess, tempfile, sys, time

AFTER_EFFECTS_APP = os.environ.get("AE_APP_NAME", "Adobe After Effects 2025")

# --- Paths (edit if needed) ---
PROJECT     = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Baseball-Animation-3.aep"
CSV_PATH    = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/teams_rgb_correct.csv"
COMMON_JSX  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/gl_common.jsxinc"

# JSX below
JSX_BATCH   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/batch_RunScored_3.jsx"

# AE / template
COMP        = os.environ.get("AE_COMP", "RunScored-384")
RS_TEMPLATE = "Best Settings"
OM_TEMPLATE = "PNG Sequence"

# RunScored image directory (files: RunScoredText-1.png .. -4.png)
RSTEXT_DIR  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/BaseBall/RunScoredText"
RS_BASENAME = "RunScoredText"  # override via AE_RS_BASENAME if needed

# Output
OUTDIR        = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/RunScored_Animation_3_384"
anim_tail    = "_3_6"
PATH_TEMPLATE = "{league}"

# League (CLI override optional)
LEAGUE = sys.argv[1] if len(sys.argv) > 1 else "MLB"

# Controls
AE_DEBUG     = "1"
PURGE_BEFORE = "1"
NO_RENDER    = "0"
QUIT_APP     = "0"    # keep AE open across runs

# Logos
LOGO_DIR  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Large/"
LOGO_TPL  = "{league}/{abbr}"
LOGO_EXTS = "png,jpg,jpeg,svg,ai,psd"

# Layer names (override via env if different)
LAYER_LOGO_PRECOMP       = "LogoPreComp"
SUBLAYER_LOGO_IN_PRECOMP = "Logo"
LAYER_LOGO_FLAT          = "TeamLogo"           # optional, plain logo on root

# Your new precomp name on root
RS_PRECOMP_NAME          = "RunScoredText-PreComp"

def _write_bridge_jsx(env):
    def esc(s): return str(s).replace("\\","\\\\").replace('"','\\"')
    lines = [ '$.setenv("'+esc(k)+'","'+esc(v)+'");' for k,v in env.items() ]
    lines.append('$.evalFile(File("'+esc(JSX_BATCH)+'"));')
    code = "\n".join(lines) + "\n"
    tmp = tempfile.NamedTemporaryFile(prefix="ae_bridge_", suffix=".jsx", delete=False)
    tmp.write(code.encode("utf-8")); tmp.flush(); tmp.close()
    return tmp.name

def _osascript(script_text):
    proc = subprocess.run(["osascript", "-s", "o"], input=script_text, text=True, capture_output=True)
    out = (proc.stdout or "").strip()
    err = (proc.stderr or "").strip()
    if err:
        print("\n[osascript stderr]\n" + err, file=sys.stderr)
    proc.check_returncode()
    return out, err

def _ping_ae(app_name):
    try:
        _osascript(f'tell application "{app_name}" to return name')
        return True
    except subprocess.CalledProcessError:
        return False

def run_after_effects(env):
    os.makedirs(OUTDIR, exist_ok=True)
    bridge = _write_bridge_jsx(env)
    osa_src = f'''
set bridgePosix to "{bridge}"
set bridgeFile to POSIX file bridgePosix
with timeout of 86400 seconds
  tell application "{AFTER_EFFECTS_APP}"
    activate
    try
      set res to DoScriptFile bridgeFile
      return "AERUN:OK:" & (res as text)
    on error errMsg number errNum
      return "AERUN:ERR:" & errNum & ":" & errMsg
    end try
  end tell
end timeout
'''
    _osascript(osa_src)

def make_env(anim_name, runnum):
    return {
        # core
        "AE_PROJECT": PROJECT,
        "AE_CSV": CSV_PATH,
        "AE_COMP": COMP,
        "AE_COMMON_JSX": COMMON_JSX,

        # logos
        "AE_LAYER_LOGO_PRECOMP": LAYER_LOGO_PRECOMP,
        "AE_SUBLAYER_LOGO_IN_PRECOMP": SUBLAYER_LOGO_IN_PRECOMP,
        "AE_LAYER_LOGO2": LAYER_LOGO_FLAT,
        "AE_LOGO_DIR": LOGO_DIR,
        "AE_LOGO_PATH_TEMPLATE": LOGO_TPL,
        "AE_LOGO_EXTS": LOGO_EXTS,

        # RunScored precomp handling
        "AE_RS_PRECOMP_NAME": RS_PRECOMP_NAME,
        "AE_RSTEXT_DIR": RSTEXT_DIR,
        "AE_RS_BASENAME": RS_BASENAME,     # "RunScoredText"
        "AE_RS_INNER_PREFIX": "RunScoredText",  # target only layers starting with this; blank = ALL footage
        "AE_RUNNUM": str(runnum),

        # output & templates
        "AE_OUTDIR": OUTDIR,
        "AE_PATH_TEMPLATE": PATH_TEMPLATE,
        "AE_ANIM": anim_name,          # "R-1" etc
        "AE_RS_TEMPLATE": RS_TEMPLATE,
        "AE_OM_TEMPLATE": OM_TEMPLATE,

        # controls
        "AE_PURGE_BEFORE_RENDER": PURGE_BEFORE,
        "AE_NO_RENDER": NO_RENDER,
        "AE_QUIT": QUIT_APP,
        "AE_LEAGUE": LEAGUE,
        "AE_DEBUG": AE_DEBUG,
    }

if __name__ == "__main__":
    print("=== After Effects RunScored Batch (precomp) ===")
    if not _ping_ae(AFTER_EFFECTS_APP):
        print(f'!! Could not talk to "{AFTER_EFFECTS_APP}"', file=sys.stderr)
        sys.exit(2)

    for runnum in range(1, 5):  # R-1..R-4
        anim_name = f"R-{runnum}{anim_tail}"
        print(f"\n== RunNum {runnum} :: AE_ANIM = {anim_name}")
        run_after_effects(make_env(anim_name, runnum))
        time.sleep(2)
    print("\nDone.")
